<?php
/**
 * Sistema de Autenticação
 * - Login para MASTER e empresas
 * - Sessões seguras com company_id
 * - Middleware de isolamento
 */

require_once __DIR__ . '/db.php';
require_once __DIR__ . '/config.php';

session_start();

// ===================== FUNÇÕES DE AUTENTICAÇÃO =====================

function login($email, $password) {
    $pdo = getPdo();
    $config = require __DIR__ . '/config.php';
    
    // Buscar usuário por email
    $stmt = $pdo->prepare("
        SELECT u.id, u.company_id, u.email, u.username, u.password_hash, u.role, u.is_active,
               c.name as company_name, c.business_type
        FROM users u
        LEFT JOIN companies c ON u.company_id = c.id
        WHERE u.email = ? AND u.is_active = 1
    ");
    $stmt->execute([$email]);
    $user = $stmt->fetch();
    
    if (!$user || !password_verify($password, $user['password_hash'])) {
        return ['success' => false, 'message' => 'Credenciais inválidas'];
    }
    
    // Criar sessão segura
    $_SESSION['user_id'] = $user['id'];
    $_SESSION['email'] = $user['email'];
    $_SESSION['role'] = $user['role'];
    $_SESSION['company_id'] = $user['company_id'];
    $_SESSION['company_name'] = $user['company_name'];
    $_SESSION['business_type'] = $user['business_type'];
    
    // Regenerar ID da sessão por segurança
    session_regenerate_id(true);
    
    return [
        'success' => true, 
        'role' => $user['role'],
        'company_id' => $user['company_id'],
        'redirect' => $user['role'] === 'master' ? '/master' : '/dashboard'
    ];
}

function logout() {
    session_destroy();
    return ['success' => true];
}

function requireAuth() {
    if (!isset($_SESSION['user_id'])) {
        http_response_code(401);
        header('Location: /login');
        exit;
    }
    return true;
}

function requireMaster() {
    requireAuth();
    if ($_SESSION['role'] !== 'master') {
        http_response_code(403);
        die('Acesso negado - apenas MASTER');
    }
    return true;
}

function requireCompanyAccess($companyId = null) {
    requireAuth();
    
    // MASTER tem acesso total
    if ($_SESSION['role'] === 'master') {
        return true;
    }
    
    // Outros roles: apenas sua empresa
    if ($companyId && $companyId != $_SESSION['company_id']) {
        http_response_code(403);
        die('Acesso negado - empresa diferente');
    }
    
    return true;
}

// ===================== MIDDLEWARE DE ISOLAMENTO =====================

function getCompanyFilter() {
    if ($_SESSION['role'] === 'master') {
        return null; // Sem filtro para MASTER
    }
    return $_SESSION['company_id'];
}

function addCompanyFilter($query, $params = []) {
    $companyId = getCompanyFilter();
    if ($companyId === null) {
        return [$query, $params];
    }
    
    // Adicionar WHERE company_id = ? se não existir
    if (stripos($query, 'WHERE') !== false) {
        $query = str_replace('WHERE', 'WHERE company_id = ? AND', $query);
    } else {
        $query .= ' WHERE company_id = ?';
    }
    
    array_unshift($params, $companyId);
    return [$query, $params];
}

// ===================== PROCESSAMENTO DE REQUESTS =====================

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    switch ($action) {
        case 'login':
            $email = $_POST['email'] ?? '';
            $password = $_POST['password'] ?? '';
            
            if (empty($email) || empty($password)) {
                echo json_encode(['success' => false, 'message' => 'Email e senha são obrigatórios']);
                exit;
            }
            
            $result = login($email, $password);
            echo json_encode($result);
            exit;
            
        case 'logout':
            $result = logout();
            echo json_encode($result);
            exit;
    }
}

// ===================== FUNÇÕES AUXILIARES =====================

function getCurrentUser() {
    if (!isset($_SESSION['user_id'])) {
        return null;
    }
    
    return [
        'id' => $_SESSION['user_id'],
        'email' => $_SESSION['email'],
        'role' => $_SESSION['role'],
        'company_id' => $_SESSION['company_id'],
        'company_name' => $_SESSION['company_name'],
        'business_type' => $_SESSION['business_type']
    ];
}

function isLoggedIn() {
    return isset($_SESSION['user_id']);
}

function isMaster() {
    return isset($_SESSION['role']) && $_SESSION['role'] === 'master';
}

function getCompanyId() {
    return $_SESSION['company_id'] ?? null;
}

?>
