<?php
/**
 * Dashboard Principal
 * - Visão geral da empresa
 * - Métricas e estatísticas
 * - Acesso rápido às funcionalidades
 */

require_once __DIR__ . '/auth.php';

// Verificar autenticação
requireAuth();

$user = getCurrentUser();
$isMaster = isMaster();

// ===================== DADOS DO DASHBOARD =====================

function getDashboardStats($companyId) {
    $pdo = getPdo();
    
    // Se for MASTER, mostrar stats globais
    if ($companyId === null) {
        $stats = [
            'total_companies' => $pdo->query("SELECT COUNT(*) FROM companies WHERE status = 'active'")->fetchColumn(),
            'total_users' => $pdo->query("SELECT COUNT(*) FROM users WHERE is_active = 1")->fetchColumn(),
            'total_appointments' => $pdo->query("SELECT COUNT(*) FROM appointments WHERE DATE(created_at) = CURDATE()")->fetchColumn(),
            'total_conversations' => $pdo->query("SELECT COUNT(*) FROM conversations WHERE DATE(created_at) = CURDATE()")->fetchColumn(),
            'revenue_today' => $pdo->query("SELECT COALESCE(SUM(paid_amount), 0) FROM appointments WHERE DATE(created_at) = CURDATE() AND payment_status = 'paid'")->fetchColumn()
        ];
    } else {
        // Stats da empresa específica
        $stmt = $pdo->prepare("
            SELECT 
                (SELECT COUNT(*) FROM doctors WHERE company_id = ? AND status = 'active') as total_doctors,
                (SELECT COUNT(*) FROM appointments WHERE company_id = ? AND DATE(created_at) = CURDATE()) as today_appointments,
                (SELECT COUNT(*) FROM appointments WHERE company_id = ? AND status = 'pending') as pending_appointments,
                (SELECT COUNT(*) FROM conversations WHERE company_id = ? AND DATE(created_at) = CURDATE()) as today_conversations,
                (SELECT COUNT(*) FROM patients WHERE company_id = ? AND status = 'active') as total_patients,
                (SELECT COALESCE(SUM(paid_amount), 0) FROM appointments WHERE company_id = ? AND DATE(created_at) = CURDATE() AND payment_status = 'paid') as revenue_today
        ");
        $stmt->execute([$companyId, $companyId, $companyId, $companyId, $companyId, $companyId]);
        $stats = $stmt->fetch();
    }
    
    return $stats;
}

function getRecentAppointments($companyId, $limit = 5) {
    $pdo = getPdo();
    
    if ($companyId === null) {
        // MASTER: mostrar de todas as empresas
        $stmt = $pdo->prepare("
            SELECT a.*, c.name as company_name, d.name as doctor_name, p.name as patient_name
            FROM appointments a
            JOIN companies c ON a.company_id = c.id
            JOIN doctors d ON a.doctor_id = d.id
            LEFT JOIN patients p ON a.patient_id = p.id
            ORDER BY a.created_at DESC
            LIMIT ?
        ");
        $stmt->execute([$limit]);
    } else {
        // Empresa específica
        $stmt = $pdo->prepare("
            SELECT a.*, d.name as doctor_name, p.name as patient_name
            FROM appointments a
            JOIN doctors d ON a.doctor_id = d.id
            LEFT JOIN patients p ON a.patient_id = p.id
            WHERE a.company_id = ?
            ORDER BY a.created_at DESC
            LIMIT ?
        ");
        $stmt->execute([$companyId, $limit]);
    }
    
    return $stmt->fetchAll();
}

function getRecentConversations($companyId, $limit = 5) {
    $pdo = getPdo();
    
    if ($companyId === null) {
        // MASTER: mostrar de todas as empresas
        $stmt = $pdo->prepare("
            SELECT conv.*, c.name as company_name
            FROM conversations conv
            JOIN companies c ON conv.company_id = c.id
            ORDER BY conv.last_message_at DESC
            LIMIT ?
        ");
        $stmt->execute([$limit]);
    } else {
        // Empresa específica
        $stmt = $pdo->prepare("
            SELECT * FROM conversations 
            WHERE company_id = ?
            ORDER BY last_message_at DESC
            LIMIT ?
        ");
        $stmt->execute([$companyId, $limit]);
    }
    
    return $stmt->fetchAll();
}

$stats = getDashboardStats($user['company_id']);
$recentAppointments = getRecentAppointments($user['company_id']);
$recentConversations = getRecentConversations($user['company_id']);

?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard - CRM Sistema</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script src="https://unpkg.com/alpinejs@3.x.x/dist/cdn.min.js" defer></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
</head>
<body class="bg-gray-50" x-data="dashboard()">
    
    <!-- Sidebar -->
    <div class="fixed inset-y-0 left-0 z-50 w-64 bg-white shadow-lg transform transition-transform duration-300 ease-in-out" 
         :class="sidebarOpen ? 'translate-x-0' : '-translate-x-full lg:translate-x-0">
        
        <!-- Logo -->
        <div class="flex items-center justify-center h-16 bg-gradient-to-r from-blue-600 to-indigo-600">
            <i class="fas fa-chart-line text-white text-2xl mr-2"></i>
            <span class="text-white font-bold text-xl">CRM Sistema</span>
        </div>

        <!-- Navigation -->
        <nav class="mt-8">
            <div class="px-4 mb-4">
                <p class="text-xs font-semibold text-gray-500 uppercase tracking-wider">Principal</p>
            </div>
            
            <a href="/dashboard.php" class="flex items-center px-4 py-3 text-blue-600 bg-blue-50 border-r-2 border-blue-600">
                <i class="fas fa-home mr-3"></i>
                Dashboard
            </a>
            
            <?php if ($isMaster): ?>
            <a href="/master.php" class="flex items-center px-4 py-3 text-gray-700 hover:bg-gray-100">
                <i class="fas fa-crown mr-3"></i>
                Painel Master
            </a>
            <?php endif; ?>
            
            <a href="/doctors.php" class="flex items-center px-4 py-3 text-gray-700 hover:bg-gray-100">
                <i class="fas fa-user-md mr-3"></i>
                Médicos
            </a>
            
            <a href="/patients.php" class="flex items-center px-4 py-3 text-gray-700 hover:bg-gray-100">
                <i class="fas fa-users mr-3"></i>
                Pacientes
            </a>
            
            <a href="/appointments.php" class="flex items-center px-4 py-3 text-gray-700 hover:bg-gray-100">
                <i class="fas fa-calendar-alt mr-3"></i>
                Agendamentos
            </a>
            
            <a href="/conversations.php" class="flex items-center px-4 py-3 text-gray-700 hover:bg-gray-100">
                <i class="fas fa-comments mr-3"></i>
                Conversas
            </a>
            
            <a href="/reports.php" class="flex items-center px-4 py-3 text-gray-700 hover:bg-gray-100">
                <i class="fas fa-chart-bar mr-3"></i>
                Relatórios
            </a>
            
            <a href="/settings.php" class="flex items-center px-4 py-3 text-gray-700 hover:bg-gray-100">
                <i class="fas fa-cog mr-3"></i>
                Configurações
            </a>
        </nav>

        <!-- User Info -->
        <div class="absolute bottom-0 w-full p-4 border-t">
            <div class="flex items-center">
                <div class="w-8 h-8 bg-blue-600 rounded-full flex items-center justify-center">
                    <i class="fas fa-user text-white text-sm"></i>
                </div>
                <div class="ml-3">
                    <p class="text-sm font-medium text-gray-900"><?= htmlspecialchars($user['email']) ?></p>
                    <p class="text-xs text-gray-500"><?= htmlspecialchars($user['company_name'] ?? 'Master') ?></p>
                </div>
            </div>
            <button @click="logout()" class="w-full mt-3 text-left text-sm text-gray-600 hover:text-gray-900">
                <i class="fas fa-sign-out-alt mr-2"></i>Sair
            </button>
        </div>
    </div>

    <!-- Main Content -->
    <div class="lg:ml-64">
        <!-- Top Bar -->
        <header class="bg-white shadow-sm border-b">
            <div class="flex items-center justify-between px-6 py-4">
                <div class="flex items-center">
                    <button @click="sidebarOpen = !sidebarOpen" class="lg:hidden mr-4">
                        <i class="fas fa-bars text-gray-600"></i>
                    </button>
                    <h1 class="text-2xl font-bold text-gray-900">Dashboard</h1>
                </div>
                
                <div class="flex items-center space-x-4">
                    <!-- Notifications -->
                    <button class="relative p-2 text-gray-600 hover:text-gray-900">
                        <i class="fas fa-bell"></i>
                        <span class="absolute -top-1 -right-1 bg-red-500 text-white text-xs rounded-full h-5 w-5 flex items-center justify-center">3</span>
                    </button>
                    
                    <!-- User Menu -->
                    <div class="relative" x-data="{ open: false }">
                        <button @click="open = !open" class="flex items-center space-x-2 text-gray-700 hover:text-gray-900">
                            <div class="w-8 h-8 bg-blue-600 rounded-full flex items-center justify-center">
                                <i class="fas fa-user text-white text-sm"></i>
                            </div>
                            <span class="hidden md:block"><?= htmlspecialchars($user['email']) ?></span>
                            <i class="fas fa-chevron-down text-xs"></i>
                        </button>
                        
                        <div x-show="open" @click.away="open = false" 
                             class="absolute right-0 mt-2 w-48 bg-white rounded-md shadow-lg py-1 z-50">
                            <a href="/profile.php" class="block px-4 py-2 text-sm text-gray-700 hover:bg-gray-100">
                                <i class="fas fa-user mr-2"></i>Perfil
                            </a>
                            <a href="/settings.php" class="block px-4 py-2 text-sm text-gray-700 hover:bg-gray-100">
                                <i class="fas fa-cog mr-2"></i>Configurações
                            </a>
                            <hr class="my-1">
                            <button @click="logout()" class="block w-full text-left px-4 py-2 text-sm text-gray-700 hover:bg-gray-100">
                                <i class="fas fa-sign-out-alt mr-2"></i>Sair
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </header>

        <!-- Dashboard Content -->
        <main class="p-6">
            <!-- Welcome Message -->
            <div class="mb-8">
                <h2 class="text-3xl font-bold text-gray-900 mb-2">
                    Olá, <?= htmlspecialchars($user['email']) ?>! 👋
                </h2>
                <p class="text-gray-600">
                    <?php if ($isMaster): ?>
                        Bem-vindo ao painel master. Aqui você pode gerenciar todas as empresas.
                    <?php else: ?>
                        Bem-vindo ao painel da <?= htmlspecialchars($user['company_name']) ?>. 
                        Aqui está um resumo das suas atividades.
                    <?php endif; ?>
                </p>
            </div>

            <!-- Stats Cards -->
            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
                <?php if ($isMaster): ?>
                <!-- Master Stats -->
                <div class="bg-white rounded-lg shadow-lg p-6 border-l-4 border-blue-500">
                    <div class="flex items-center">
                        <div class="p-3 rounded-full bg-blue-100 text-blue-600">
                            <i class="fas fa-building text-xl"></i>
                        </div>
                        <div class="ml-4">
                            <p class="text-sm font-medium text-gray-600">Empresas Ativas</p>
                            <p class="text-2xl font-semibold text-gray-900"><?= $stats['total_companies'] ?></p>
                        </div>
                    </div>
                </div>
                
                <div class="bg-white rounded-lg shadow-lg p-6 border-l-4 border-green-500">
                    <div class="flex items-center">
                        <div class="p-3 rounded-full bg-green-100 text-green-600">
                            <i class="fas fa-users text-xl"></i>
                        </div>
                        <div class="ml-4">
                            <p class="text-sm font-medium text-gray-600">Usuários Ativos</p>
                            <p class="text-2xl font-semibold text-gray-900"><?= $stats['total_users'] ?></p>
                        </div>
                    </div>
                </div>
                
                <div class="bg-white rounded-lg shadow-lg p-6 border-l-4 border-purple-500">
                    <div class="flex items-center">
                        <div class="p-3 rounded-full bg-purple-100 text-purple-600">
                            <i class="fas fa-calendar-day text-xl"></i>
                        </div>
                        <div class="ml-4">
                            <p class="text-sm font-medium text-gray-600">Agendamentos Hoje</p>
                            <p class="text-2xl font-semibold text-gray-900"><?= $stats['total_appointments'] ?></p>
                        </div>
                    </div>
                </div>
                
                <div class="bg-white rounded-lg shadow-lg p-6 border-l-4 border-yellow-500">
                    <div class="flex items-center">
                        <div class="p-3 rounded-full bg-yellow-100 text-yellow-600">
                            <i class="fas fa-dollar-sign text-xl"></i>
                        </div>
                        <div class="ml-4">
                            <p class="text-sm font-medium text-gray-600">Receita Hoje</p>
                            <p class="text-2xl font-semibold text-gray-900">R$ <?= number_format($stats['revenue_today'], 2, ',', '.') ?></p>
                        </div>
                    </div>
                </div>
                <?php else: ?>
                <!-- Company Stats -->
                <div class="bg-white rounded-lg shadow-lg p-6 border-l-4 border-blue-500">
                    <div class="flex items-center">
                        <div class="p-3 rounded-full bg-blue-100 text-blue-600">
                            <i class="fas fa-user-md text-xl"></i>
                        </div>
                        <div class="ml-4">
                            <p class="text-sm font-medium text-gray-600">Médicos Ativos</p>
                            <p class="text-2xl font-semibold text-gray-900"><?= $stats['total_doctors'] ?></p>
                        </div>
                    </div>
                </div>
                
                <div class="bg-white rounded-lg shadow-lg p-6 border-l-4 border-green-500">
                    <div class="flex items-center">
                        <div class="p-3 rounded-full bg-green-100 text-green-600">
                            <i class="fas fa-users text-xl"></i>
                        </div>
                        <div class="ml-4">
                            <p class="text-sm font-medium text-gray-600">Pacientes</p>
                            <p class="text-2xl font-semibold text-gray-900"><?= $stats['total_patients'] ?></p>
                        </div>
                    </div>
                </div>
                
                <div class="bg-white rounded-lg shadow-lg p-6 border-l-4 border-yellow-500">
                    <div class="flex items-center">
                        <div class="p-3 rounded-full bg-yellow-100 text-yellow-600">
                            <i class="fas fa-clock text-xl"></i>
                        </div>
                        <div class="ml-4">
                            <p class="text-sm font-medium text-gray-600">Pendentes</p>
                            <p class="text-2xl font-semibold text-gray-900"><?= $stats['pending_appointments'] ?></p>
                        </div>
                    </div>
                </div>
                
                <div class="bg-white rounded-lg shadow-lg p-6 border-l-4 border-purple-500">
                    <div class="flex items-center">
                        <div class="p-3 rounded-full bg-purple-100 text-purple-600">
                            <i class="fas fa-dollar-sign text-xl"></i>
                        </div>
                        <div class="ml-4">
                            <p class="text-sm font-medium text-gray-600">Receita Hoje</p>
                            <p class="text-2xl font-semibold text-gray-900">R$ <?= number_format($stats['revenue_today'], 2, ',', '.') ?></p>
                        </div>
                    </div>
                </div>
                <?php endif; ?>
            </div>

            <!-- Recent Activity -->
            <div class="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-8">
                <!-- Recent Appointments -->
                <div class="bg-white rounded-lg shadow-lg">
                    <div class="px-6 py-4 border-b border-gray-200 bg-gradient-to-r from-gray-50 to-gray-100">
                        <div class="flex items-center justify-between">
                            <h3 class="text-lg font-medium text-gray-900">Agendamentos Recentes</h3>
                            <a href="/appointments.php" class="text-blue-600 hover:text-blue-700 text-sm font-medium">
                                Ver todos
                            </a>
                        </div>
                    </div>
                    <div class="p-6">
                        <?php if (empty($recentAppointments)): ?>
                        <div class="text-center py-8">
                            <i class="fas fa-calendar-times text-gray-400 text-4xl mb-4"></i>
                            <p class="text-gray-500">Nenhum agendamento recente</p>
                        </div>
                        <?php else: ?>
                        <div class="space-y-4">
                            <?php foreach ($recentAppointments as $appointment): ?>
                            <div class="flex items-center justify-between p-4 bg-gray-50 rounded-lg">
                                <div class="flex items-center">
                                    <div class="w-10 h-10 bg-blue-100 rounded-full flex items-center justify-center">
                                        <i class="fas fa-calendar text-blue-600"></i>
                                    </div>
                                    <div class="ml-4">
                                        <p class="font-medium text-gray-900">
                                            <?= htmlspecialchars($appointment['patient_name'] ?: $appointment['patient_name'] ?: 'Sem nome') ?>
                                        </p>
                                        <p class="text-sm text-gray-600">
                                            <?= htmlspecialchars($appointment['doctor_name']) ?> - 
                                            <?= date('d/m/Y H:i', strtotime($appointment['scheduled_at'])) ?>
                                        </p>
                                        <?php if ($isMaster): ?>
                                        <p class="text-xs text-gray-500">
                                            <?= htmlspecialchars($appointment['company_name']) ?>
                                        </p>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                <div class="flex items-center space-x-2">
                                    <span class="px-2 py-1 text-xs font-medium rounded-full
                                        <?php 
                                        switch($appointment['status']) {
                                            case 'pending': echo 'bg-yellow-100 text-yellow-800'; break;
                                            case 'confirmed': echo 'bg-green-100 text-green-800'; break;
                                            case 'completed': echo 'bg-blue-100 text-blue-800'; break;
                                            case 'cancelled': echo 'bg-red-100 text-red-800'; break;
                                            case 'no_show': echo 'bg-gray-100 text-gray-800'; break;
                                        }
                                        ?>">
                                        <?= ucfirst($appointment['status']) ?>
                                    </span>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Recent Conversations -->
                <div class="bg-white rounded-lg shadow-lg">
                    <div class="px-6 py-4 border-b border-gray-200 bg-gradient-to-r from-gray-50 to-gray-100">
                        <div class="flex items-center justify-between">
                            <h3 class="text-lg font-medium text-gray-900">Conversas Recentes</h3>
                            <a href="/conversations.php" class="text-blue-600 hover:text-blue-700 text-sm font-medium">
                                Ver todas
                            </a>
                        </div>
                    </div>
                    <div class="p-6">
                        <?php if (empty($recentConversations)): ?>
                        <div class="text-center py-8">
                            <i class="fas fa-comments text-gray-400 text-4xl mb-4"></i>
                            <p class="text-gray-500">Nenhuma conversa recente</p>
                        </div>
                        <?php else: ?>
                        <div class="space-y-4">
                            <?php foreach ($recentConversations as $conversation): ?>
                            <div class="flex items-center justify-between p-4 bg-gray-50 rounded-lg">
                                <div class="flex items-center">
                                    <div class="w-10 h-10 bg-green-100 rounded-full flex items-center justify-center">
                                        <i class="fab fa-whatsapp text-green-600"></i>
                                    </div>
                                    <div class="ml-4">
                                        <p class="font-medium text-gray-900">
                                            <?= htmlspecialchars($conversation['contact_name'] ?: $conversation['phone_number']) ?>
                                        </p>
                                        <p class="text-sm text-gray-600">
                                            <?= $conversation['last_message_at'] ? date('d/m/Y H:i', strtotime($conversation['last_message_at'])) : 'Sem mensagens' ?>
                                        </p>
                                        <?php if ($isMaster): ?>
                                        <p class="text-xs text-gray-500">
                                            <?= htmlspecialchars($conversation['company_name']) ?>
                                        </p>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                <div class="flex items-center space-x-2">
                                    <span class="px-2 py-1 text-xs font-medium rounded-full
                                        <?php 
                                        switch($conversation['status']) {
                                            case 'active': echo 'bg-green-100 text-green-800'; break;
                                            case 'closed': echo 'bg-gray-100 text-gray-800'; break;
                                            case 'blocked': echo 'bg-red-100 text-red-800'; break;
                                        }
                                        ?>">
                                        <?= ucfirst($conversation['status']) ?>
                                    </span>
                                    <?php if ($conversation['unread_messages'] > 0): ?>
                                    <span class="bg-red-500 text-white text-xs rounded-full h-5 w-5 flex items-center justify-center">
                                        <?= $conversation['unread_messages'] ?>
                                    </span>
                                    <?php endif; ?>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

            <!-- Quick Actions -->
            <div class="bg-white rounded-lg shadow-lg">
                <div class="px-6 py-4 border-b border-gray-200">
                    <h3 class="text-lg font-medium text-gray-900">Ações Rápidas</h3>
                </div>
                <div class="p-6">
                    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
                        <a href="/appointments.php?action=new" class="flex items-center p-4 bg-blue-50 rounded-lg hover:bg-blue-100 transition-colors">
                            <i class="fas fa-plus-circle text-blue-600 text-xl mr-3"></i>
                            <div>
                                <p class="font-medium text-gray-900">Novo Agendamento</p>
                                <p class="text-sm text-gray-600">Agendar consulta</p>
                            </div>
                        </a>
                        
                        <a href="/patients.php?action=new" class="flex items-center p-4 bg-green-50 rounded-lg hover:bg-green-100 transition-colors">
                            <i class="fas fa-user-plus text-green-600 text-xl mr-3"></i>
                            <div>
                                <p class="font-medium text-gray-900">Novo Paciente</p>
                                <p class="text-sm text-gray-600">Cadastrar paciente</p>
                            </div>
                        </a>
                        
                        <a href="/doctors.php?action=new" class="flex items-center p-4 bg-purple-50 rounded-lg hover:bg-purple-100 transition-colors">
                            <i class="fas fa-user-md text-purple-600 text-xl mr-3"></i>
                            <div>
                                <p class="font-medium text-gray-900">Novo Médico</p>
                                <p class="text-sm text-gray-600">Cadastrar médico</p>
                            </div>
                        </a>
                        
                        <a href="/reports.php" class="flex items-center p-4 bg-yellow-50 rounded-lg hover:bg-yellow-100 transition-colors">
                            <i class="fas fa-chart-bar text-yellow-600 text-xl mr-3"></i>
                            <div>
                                <p class="font-medium text-gray-900">Relatórios</p>
                                <p class="text-sm text-gray-600">Ver relatórios</p>
                            </div>
                        </a>
                    </div>
                </div>
            </div>
        </main>
    </div>

    <!-- Overlay para mobile -->
    <div x-show="sidebarOpen" @click="sidebarOpen = false" 
         class="fixed inset-0 z-40 bg-black bg-opacity-50 lg:hidden" 
         x-transition:enter="transition-opacity ease-linear duration-300"
         x-transition:enter-start="opacity-0"
         x-transition:enter-end="opacity-100"
         x-transition:leave="transition-opacity ease-linear duration-300"
         x-transition:leave-start="opacity-100"
         x-transition:leave-end="opacity-0"></div>

    <script>
        function dashboard() {
            return {
                sidebarOpen: false,

                async logout() {
                    try {
                        const formData = new FormData();
                        formData.append('action', 'logout');
                        
                        const response = await fetch('/auth.php', {
                            method: 'POST',
                            body: formData
                        });
                        
                        const result = await response.json();
                        
                        if (result.success) {
                            window.location.href = '/login.php';
                        }
                    } catch (error) {
                        console.error('Logout error:', error);
                        window.location.href = '/login.php';
                    }
                }
            }
        }
    </script>

</body>
</html>
