<?php
/**
 * Painel Master Profissional
 * - Gerenciar empresas completas
 * - Criar usuários avançados
 * - Visão global do sistema
 */

require_once __DIR__ . '/auth.php';

// Verificar se é MASTER
requireMaster();

$user = getCurrentUser();

// ===================== FUNÇÕES MASTER PROFISSIONAIS =====================

function getAllCompanies() {
    $pdo = getPdo();
    $stmt = $pdo->query("
        SELECT c.*, 
               (SELECT COUNT(*) FROM users WHERE company_id = c.id AND is_active = 1) as user_count,
               (SELECT COUNT(*) FROM doctors WHERE company_id = c.id AND is_active = 1) as doctor_count,
               (SELECT COUNT(*) FROM appointments WHERE company_id = c.id AND DATE(created_at) = CURDATE()) as today_appointments,
               (SELECT COUNT(*) FROM conversations WHERE company_id = c.id AND status = 'active') as active_conversations
        FROM companies c 
        ORDER BY c.created_at DESC
    ");
    return $stmt->fetchAll();
}

function getCompanyStats() {
    $pdo = getPdo();
    
    $stats = [
        'total_companies' => $pdo->query("SELECT COUNT(*) FROM companies")->fetchColumn(),
        'active_companies' => $pdo->query("SELECT COUNT(*) FROM companies WHERE status = 'active'")->fetchColumn(),
        'trial_companies' => $pdo->query("SELECT COUNT(*) FROM companies WHERE status = 'trial'")->fetchColumn(),
        'total_users' => $pdo->query("SELECT COUNT(*) FROM users WHERE is_active = 1")->fetchColumn(),
        'total_doctors' => $pdo->query("SELECT COUNT(*) FROM doctors WHERE is_active = 1")->fetchColumn(),
        'total_appointments' => $pdo->query("SELECT COUNT(*) FROM appointments WHERE DATE(created_at) = CURDATE()")->fetchColumn(),
        'total_conversations' => $pdo->query("SELECT COUNT(*) FROM conversations WHERE DATE(created_at) = CURDATE()")->fetchColumn(),
        'revenue_today' => $pdo->query("SELECT COALESCE(SUM(paid_amount), 0) FROM appointments WHERE DATE(created_at) = CURDATE() AND payment_status = 'paid'")->fetchColumn()
    ];
    
    return $stats;
}

function createCompany($data) {
    $pdo = getPdo();
    
    try {
        $pdo->beginTransaction();
        
        // Criar empresa com dados completos
        $stmt = $pdo->prepare("
            INSERT INTO companies (
                name, business_name, cnpj, business_type, contact_email, contact_phone, 
                whatsapp, website, address_street, address_number, address_neighborhood, 
                address_city, address_state, address_zipcode, plan_type, max_doctors, 
                max_users, max_appointments_month, status, created_by
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");
        
        $stmt->execute([
            $data['name'],
            $data['business_name'] ?? $data['name'],
            $data['cnpj'] ?? null,
            $data['business_type'],
            $data['contact_email'] ?? null,
            $data['contact_phone'] ?? null,
            $data['whatsapp'] ?? null,
            $data['website'] ?? null,
            $data['address_street'] ?? null,
            $data['address_number'] ?? null,
            $data['address_neighborhood'] ?? null,
            $data['address_city'] ?? null,
            $data['address_state'] ?? null,
            $data['address_zipcode'] ?? null,
            $data['plan_type'] ?? 'basic',
            $data['max_doctors'] ?? 5,
            $data['max_users'] ?? 10,
            $data['max_appointments_month'] ?? 1000,
            'trial',
            $user['id']
        ]);
        
        $companyId = $pdo->lastInsertId();
        
        // Criar usuário admin padrão
        $adminEmail = $data['admin_email'] ?? "admin@{$data['name']}.com";
        $adminPassword = password_hash($data['admin_password'] ?? 'admin123', PASSWORD_DEFAULT);
        
        $stmt = $pdo->prepare("
            INSERT INTO users (company_id, email, username, password_hash, first_name, last_name, role, is_active, created_by)
            VALUES (?, ?, 'admin', ?, 'Admin', 'Principal', 'admin', 1, ?)
        ");
        $stmt->execute([$companyId, $adminEmail, $adminPassword, $user['id']]);
        
        // Criar configurações padrão
        $defaultSettings = [
            ['business_hours_start', '08:00', 'string'],
            ['business_hours_end', '18:00', 'string'],
            ['appointment_duration_default', '30', 'number'],
            ['ai_enabled', 'true', 'boolean'],
            ['whatsapp_integration', 'true', 'boolean'],
            ['auto_confirm_appointments', 'false', 'boolean'],
            ['send_reminders', 'true', 'boolean'],
            ['reminder_hours_before', '24', 'number']
        ];
        
        $stmt = $pdo->prepare("
            INSERT INTO company_settings (company_id, setting_key, setting_value, setting_type)
            VALUES (?, ?, ?, ?)
        ");
        
        foreach ($defaultSettings as $setting) {
            $stmt->execute([$companyId, $setting[0], $setting[1], $setting[2]]);
        }
        
        $pdo->commit();
        
        return [
            'success' => true,
            'company_id' => $companyId,
            'admin_email' => $adminEmail,
            'admin_password' => $data['admin_password'] ?? 'admin123'
        ];
        
    } catch (Exception $e) {
        $pdo->rollBack();
        return ['success' => false, 'message' => $e->getMessage()];
    }
}

function toggleCompanyStatus($companyId, $newStatus) {
    $pdo = getPdo();
    
    try {
        $stmt = $pdo->prepare("UPDATE companies SET status = ? WHERE id = ?");
        $stmt->execute([$newStatus, $companyId]);
        
        return [
            'success' => true,
            'message' => "Status alterado para {$newStatus}"
        ];
        
    } catch (Exception $e) {
        return ['success' => false, 'message' => $e->getMessage()];
    }
}

// Processar requests
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    switch ($action) {
        case 'create_company':
            $companyData = [
                'name' => $_POST['name'] ?? '',
                'business_name' => $_POST['business_name'] ?? '',
                'cnpj' => $_POST['cnpj'] ?? '',
                'business_type' => $_POST['business_type'] ?? 'clinic',
                'contact_email' => $_POST['contact_email'] ?? '',
                'contact_phone' => $_POST['contact_phone'] ?? '',
                'whatsapp' => $_POST['whatsapp'] ?? '',
                'website' => $_POST['website'] ?? '',
                'address_street' => $_POST['address_street'] ?? '',
                'address_number' => $_POST['address_number'] ?? '',
                'address_neighborhood' => $_POST['address_neighborhood'] ?? '',
                'address_city' => $_POST['address_city'] ?? '',
                'address_state' => $_POST['address_state'] ?? '',
                'address_zipcode' => $_POST['address_zipcode'] ?? '',
                'plan_type' => $_POST['plan_type'] ?? 'basic',
                'max_doctors' => $_POST['max_doctors'] ?? 5,
                'max_users' => $_POST['max_users'] ?? 10,
                'max_appointments_month' => $_POST['max_appointments_month'] ?? 1000,
                'admin_email' => $_POST['admin_email'] ?? '',
                'admin_password' => $_POST['admin_password'] ?? 'admin123'
            ];
            
            if (empty($companyData['name']) || empty($companyData['business_type'])) {
                echo json_encode(['success' => false, 'message' => 'Nome e tipo de negócio são obrigatórios']);
                exit;
            }
            
            $result = createCompany($companyData);
            echo json_encode($result);
            exit;
            
        case 'toggle_status':
            $companyId = $_POST['company_id'] ?? '';
            $newStatus = $_POST['new_status'] ?? '';
            $result = toggleCompanyStatus($companyId, $newStatus);
            echo json_encode($result);
            exit;
    }
}

$companies = getAllCompanies();
$stats = getCompanyStats();

?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Painel Master - CRM Profissional</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script src="https://unpkg.com/alpinejs@3.x.x/dist/cdn.min.js" defer></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
</head>
<body class="bg-gray-50" x-data="masterPanel()">
    
    <!-- Sidebar -->
    <div class="fixed inset-y-0 left-0 z-50 w-64 bg-white shadow-lg">
        <!-- Logo -->
        <div class="flex items-center justify-center h-16 bg-gradient-to-r from-purple-600 to-indigo-600">
            <i class="fas fa-crown text-white text-2xl mr-2"></i>
            <span class="text-white font-bold text-xl">Master Panel</span>
        </div>

        <!-- Navigation -->
        <nav class="mt-8">
            <div class="px-4 mb-4">
                <p class="text-xs font-semibold text-gray-500 uppercase tracking-wider">Master</p>
            </div>
            
            <a href="/master.php" class="flex items-center px-4 py-3 text-purple-600 bg-purple-50 border-r-2 border-purple-600">
                <i class="fas fa-tachometer-alt mr-3"></i>
                Dashboard Master
            </a>
            
            <a href="/users.php" class="flex items-center px-4 py-3 text-gray-700 hover:bg-gray-100">
                <i class="fas fa-users mr-3"></i>
                Usuários
            </a>
            
            <a href="/companies.php" class="flex items-center px-4 py-3 text-gray-700 hover:bg-gray-100">
                <i class="fas fa-building mr-3"></i>
                Empresas
            </a>
            
            <a href="/analytics.php" class="flex items-center px-4 py-3 text-gray-700 hover:bg-gray-100">
                <i class="fas fa-chart-bar mr-3"></i>
                Analytics
            </a>
            
            <a href="/settings.php" class="flex items-center px-4 py-3 text-gray-700 hover:bg-gray-100">
                <i class="fas fa-cog mr-3"></i>
                Configurações
            </a>
        </nav>

        <!-- User Info -->
        <div class="absolute bottom-0 w-full p-4 border-t">
            <div class="flex items-center">
                <div class="w-8 h-8 bg-purple-600 rounded-full flex items-center justify-center">
                    <i class="fas fa-crown text-white text-sm"></i>
                </div>
                <div class="ml-3">
                    <p class="text-sm font-medium text-gray-900"><?= htmlspecialchars($user['email']) ?></p>
                    <p class="text-xs text-gray-500">Master Admin</p>
                </div>
            </div>
            <button @click="logout()" class="w-full mt-3 text-left text-sm text-gray-600 hover:text-gray-900">
                <i class="fas fa-sign-out-alt mr-2"></i>Sair
            </button>
        </div>
    </div>

    <!-- Main Content -->
    <div class="ml-64">
        <!-- Top Bar -->
        <header class="bg-white shadow-sm border-b">
            <div class="flex items-center justify-between px-6 py-4">
                <div>
                    <h1 class="text-2xl font-bold text-gray-900">Painel Master</h1>
                    <p class="text-gray-600">Gerenciamento global do sistema CRM</p>
                </div>
                
                <div class="flex items-center space-x-4">
                    <button @click="showCreateCompany = true" 
                            class="bg-gradient-to-r from-purple-600 to-indigo-600 text-white px-6 py-2 rounded-lg hover:from-purple-700 hover:to-indigo-700 transition-all duration-200 shadow-lg">
                        <i class="fas fa-plus mr-2"></i>Nova Empresa
                    </button>
                </div>
            </div>
        </header>

        <!-- Dashboard Content -->
        <main class="p-6">
            <!-- Stats Cards -->
            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
                <div class="bg-white rounded-lg shadow-lg p-6 border-l-4 border-blue-500">
                    <div class="flex items-center">
                        <div class="p-3 rounded-full bg-blue-100 text-blue-600">
                            <i class="fas fa-building text-xl"></i>
                        </div>
                        <div class="ml-4">
                            <p class="text-sm font-medium text-gray-600">Total Empresas</p>
                            <p class="text-2xl font-semibold text-gray-900"><?= $stats['total_companies'] ?></p>
                            <p class="text-xs text-green-600"><?= $stats['active_companies'] ?> ativas</p>
                        </div>
                    </div>
                </div>
                
                <div class="bg-white rounded-lg shadow-lg p-6 border-l-4 border-green-500">
                    <div class="flex items-center">
                        <div class="p-3 rounded-full bg-green-100 text-green-600">
                            <i class="fas fa-users text-xl"></i>
                        </div>
                        <div class="ml-4">
                            <p class="text-sm font-medium text-gray-600">Usuários Ativos</p>
                            <p class="text-2xl font-semibold text-gray-900"><?= $stats['total_users'] ?></p>
                        </div>
                    </div>
                </div>
                
                <div class="bg-white rounded-lg shadow-lg p-6 border-l-4 border-purple-500">
                    <div class="flex items-center">
                        <div class="p-3 rounded-full bg-purple-100 text-purple-600">
                            <i class="fas fa-user-md text-xl"></i>
                        </div>
                        <div class="ml-4">
                            <p class="text-sm font-medium text-gray-600">Médicos Ativos</p>
                            <p class="text-2xl font-semibold text-gray-900"><?= $stats['total_doctors'] ?></p>
                        </div>
                    </div>
                </div>
                
                <div class="bg-white rounded-lg shadow-lg p-6 border-l-4 border-yellow-500">
                    <div class="flex items-center">
                        <div class="p-3 rounded-full bg-yellow-100 text-yellow-600">
                            <i class="fas fa-calendar-day text-xl"></i>
                        </div>
                        <div class="ml-4">
                            <p class="text-sm font-medium text-gray-600">Agendamentos Hoje</p>
                            <p class="text-2xl font-semibold text-gray-900"><?= $stats['total_appointments'] ?></p>
                            <p class="text-xs text-green-600">R$ <?= number_format($stats['revenue_today'], 2, ',', '.') ?></p>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Companies List -->
            <div class="bg-white rounded-lg shadow-lg">
                <div class="px-6 py-4 border-b border-gray-200 bg-gradient-to-r from-gray-50 to-gray-100">
                    <div class="flex items-center justify-between">
                        <h3 class="text-lg font-medium text-gray-900">Empresas Cadastradas</h3>
                        <div class="flex items-center space-x-2">
                            <span class="text-sm text-gray-500">Total: <?= count($companies) ?></span>
                            <button @click="refreshCompanies()" class="text-gray-400 hover:text-gray-600">
                                <i class="fas fa-sync-alt"></i>
                            </button>
                        </div>
                    </div>
                </div>
                <div class="overflow-x-auto">
                    <table class="min-w-full divide-y divide-gray-200">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Empresa</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Tipo</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Plano</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Usuários</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Médicos</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Hoje</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Ações</th>
                            </tr>
                        </thead>
                        <tbody class="bg-white divide-y divide-gray-200">
                            <?php foreach ($companies as $company): ?>
                            <tr class="hover:bg-gray-50">
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <div class="flex items-center">
                                        <div class="w-10 h-10 bg-gradient-to-r from-blue-500 to-purple-500 rounded-lg flex items-center justify-center text-white font-bold">
                                            <?= strtoupper(substr($company['name'], 0, 2)) ?>
                                        </div>
                                        <div class="ml-4">
                                            <div class="text-sm font-medium text-gray-900"><?= htmlspecialchars($company['name']) ?></div>
                                            <?php if ($company['business_name'] && $company['business_name'] !== $company['name']): ?>
                                            <div class="text-sm text-gray-500"><?= htmlspecialchars($company['business_name']) ?></div>
                                            <?php endif; ?>
                                            <?php if ($company['cnpj']): ?>
                                            <div class="text-xs text-gray-400">CNPJ: <?= htmlspecialchars($company['cnpj']) ?></div>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <span class="px-2 py-1 text-xs font-medium bg-blue-100 text-blue-800 rounded-full">
                                        <?= ucfirst($company['business_type']) ?>
                                    </span>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <span class="px-2 py-1 text-xs font-medium rounded-full
                                        <?php 
                                        switch($company['plan_type']) {
                                            case 'basic': echo 'bg-gray-100 text-gray-800'; break;
                                            case 'professional': echo 'bg-blue-100 text-blue-800'; break;
                                            case 'enterprise': echo 'bg-purple-100 text-purple-800'; break;
                                        }
                                        ?>">
                                        <?= ucfirst($company['plan_type']) ?>
                                    </span>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                    <div class="flex items-center">
                                        <i class="fas fa-users text-gray-400 mr-1"></i>
                                        <?= $company['user_count'] ?>
                                    </div>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                    <div class="flex items-center">
                                        <i class="fas fa-user-md text-gray-400 mr-1"></i>
                                        <?= $company['doctor_count'] ?>
                                    </div>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                    <div class="flex items-center">
                                        <i class="fas fa-calendar text-gray-400 mr-1"></i>
                                        <?= $company['today_appointments'] ?>
                                    </div>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <span class="px-2 py-1 text-xs font-medium rounded-full
                                        <?php 
                                        switch($company['status']) {
                                            case 'active': echo 'bg-green-100 text-green-800'; break;
                                            case 'trial': echo 'bg-yellow-100 text-yellow-800'; break;
                                            case 'inactive': echo 'bg-red-100 text-red-800'; break;
                                            case 'suspended': echo 'bg-gray-100 text-gray-800'; break;
                                        }
                                        ?>">
                                        <?= ucfirst($company['status']) ?>
                                    </span>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                    <div class="flex items-center space-x-2">
                                        <a href="/company.php?id=<?= $company['id'] ?>" 
                                           class="text-blue-600 hover:text-blue-900" title="Ver detalhes">
                                            <i class="fas fa-eye"></i>
                                        </a>
                                        <a href="/company.php?id=<?= $company['id'] ?>&edit=1" 
                                           class="text-green-600 hover:text-green-900" title="Editar">
                                            <i class="fas fa-edit"></i>
                                        </a>
                                        <button @click="toggleCompanyStatus(<?= $company['id'] ?>, '<?= $company['status'] === 'active' ? 'inactive' : 'active' ?>')" 
                                                class="text-red-600 hover:text-red-900" title="Alterar status">
                                            <i class="fas fa-<?= $company['status'] === 'active' ? 'pause' : 'play' ?>"></i>
                                        </button>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </main>
    </div>

    <!-- Modal Criar Empresa -->
    <div x-show="showCreateCompany" 
         x-transition:enter="transition-opacity ease-linear duration-300"
         x-transition:enter-start="opacity-0"
         x-transition:enter-end="opacity-100"
         x-transition:leave="transition-opacity ease-linear duration-300"
         x-transition:leave-start="opacity-100"
         x-transition:leave-end="opacity-0"
         class="fixed inset-0 z-50 overflow-y-auto" 
         style="display: none;">
        
        <div class="flex items-center justify-center min-h-screen pt-4 px-4 pb-20 text-center sm:block sm:p-0">
            <div class="fixed inset-0 bg-gray-500 bg-opacity-75 transition-opacity" @click="showCreateCompany = false"></div>

            <div class="inline-block align-bottom bg-white rounded-lg text-left overflow-hidden shadow-xl transform transition-all sm:my-8 sm:align-middle sm:max-w-4xl sm:w-full">
                <form @submit.prevent="createCompany">
                    <div class="bg-white px-4 pt-5 pb-4 sm:p-6 sm:pb-4">
                        <div class="sm:flex sm:items-start">
                            <div class="mx-auto flex-shrink-0 flex items-center justify-center h-12 w-12 rounded-full bg-purple-100 sm:mx-0 sm:h-10 sm:w-10">
                                <i class="fas fa-building text-purple-600"></i>
                            </div>
                            <div class="mt-3 text-center sm:mt-0 sm:ml-4 sm:text-left w-full">
                                <h3 class="text-lg leading-6 font-medium text-gray-900 mb-6">Nova Empresa</h3>
                                
                                <!-- Tabs -->
                                <div class="border-b border-gray-200 mb-6">
                                    <nav class="-mb-px flex space-x-8">
                                        <button type="button" @click="activeTab = 'basic'" 
                                                :class="activeTab === 'basic' ? 'border-purple-500 text-purple-600' : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'"
                                                class="whitespace-nowrap py-2 px-1 border-b-2 font-medium text-sm">
                                            Dados Básicos
                                        </button>
                                        <button type="button" @click="activeTab = 'contact'" 
                                                :class="activeTab === 'contact' ? 'border-purple-500 text-purple-600' : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'"
                                                class="whitespace-nowrap py-2 px-1 border-b-2 font-medium text-sm">
                                            Contato
                                        </button>
                                        <button type="button" @click="activeTab = 'address'" 
                                                :class="activeTab === 'address' ? 'border-purple-500 text-purple-600' : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'"
                                                class="whitespace-nowrap py-2 px-1 border-b-2 font-medium text-sm">
                                            Endereço
                                        </button>
                                        <button type="button" @click="activeTab = 'config'" 
                                                :class="activeTab === 'config' ? 'border-purple-500 text-purple-600' : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'"
                                                class="whitespace-nowrap py-2 px-1 border-b-2 font-medium text-sm">
                                            Configurações
                                        </button>
                                    </nav>
                                </div>
                                
                                <!-- Tab Content -->
                                <div class="space-y-6">
                                    <!-- Dados Básicos -->
                                    <div x-show="activeTab === 'basic'" class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                        <div>
                                            <label class="block text-sm font-medium text-gray-700 mb-1">Nome da Empresa *</label>
                                            <input type="text" x-model="newCompany.name" required
                                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-purple-500 focus:border-purple-500">
                                        </div>
                                        <div>
                                            <label class="block text-sm font-medium text-gray-700 mb-1">Razão Social</label>
                                            <input type="text" x-model="newCompany.business_name"
                                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-purple-500 focus:border-purple-500">
                                        </div>
                                        <div>
                                            <label class="block text-sm font-medium text-gray-700 mb-1">CNPJ</label>
                                            <input type="text" x-model="newCompany.cnpj" placeholder="00.000.000/0000-00"
                                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-purple-500 focus:border-purple-500">
                                        </div>
                                        <div>
                                            <label class="block text-sm font-medium text-gray-700 mb-1">Tipo de Negócio *</label>
                                            <select x-model="newCompany.business_type" required
                                                    class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-purple-500 focus:border-purple-500">
                                                <option value="">Selecione...</option>
                                                <option value="clinic">Clínica Médica</option>
                                                <option value="hospital">Hospital</option>
                                                <option value="dental">Clínica Odontológica</option>
                                                <option value="veterinary">Clínica Veterinária</option>
                                                <option value="psychology">Psicologia</option>
                                                <option value="physiotherapy">Fisioterapia</option>
                                                <option value="other">Outro</option>
                                            </select>
                                        </div>
                                    </div>
                                    
                                    <!-- Contato -->
                                    <div x-show="activeTab === 'contact'" class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                        <div>
                                            <label class="block text-sm font-medium text-gray-700 mb-1">Email de Contato</label>
                                            <input type="email" x-model="newCompany.contact_email"
                                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-purple-500 focus:border-purple-500">
                                        </div>
                                        <div>
                                            <label class="block text-sm font-medium text-gray-700 mb-1">Telefone</label>
                                            <input type="tel" x-model="newCompany.contact_phone"
                                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-purple-500 focus:border-purple-500">
                                        </div>
                                        <div>
                                            <label class="block text-sm font-medium text-gray-700 mb-1">WhatsApp</label>
                                            <input type="tel" x-model="newCompany.whatsapp"
                                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-purple-500 focus:border-purple-500">
                                        </div>
                                        <div>
                                            <label class="block text-sm font-medium text-gray-700 mb-1">Website</label>
                                            <input type="url" x-model="newCompany.website"
                                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-purple-500 focus:border-purple-500">
                                        </div>
                                    </div>
                                    
                                    <!-- Endereço -->
                                    <div x-show="activeTab === 'address'" class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                        <div class="md:col-span-2">
                                            <label class="block text-sm font-medium text-gray-700 mb-1">Rua</label>
                                            <input type="text" x-model="newCompany.address_street"
                                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-purple-500 focus:border-purple-500">
                                        </div>
                                        <div>
                                            <label class="block text-sm font-medium text-gray-700 mb-1">Número</label>
                                            <input type="text" x-model="newCompany.address_number"
                                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-purple-500 focus:border-purple-500">
                                        </div>
                                        <div>
                                            <label class="block text-sm font-medium text-gray-700 mb-1">Bairro</label>
                                            <input type="text" x-model="newCompany.address_neighborhood"
                                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-purple-500 focus:border-purple-500">
                                        </div>
                                        <div>
                                            <label class="block text-sm font-medium text-gray-700 mb-1">Cidade</label>
                                            <input type="text" x-model="newCompany.address_city"
                                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-purple-500 focus:border-purple-500">
                                        </div>
                                        <div>
                                            <label class="block text-sm font-medium text-gray-700 mb-1">Estado</label>
                                            <select x-model="newCompany.address_state"
                                                    class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-purple-500 focus:border-purple-500">
                                                <option value="">Selecione...</option>
                                                <option value="AC">Acre</option>
                                                <option value="AL">Alagoas</option>
                                                <option value="AP">Amapá</option>
                                                <option value="AM">Amazonas</option>
                                                <option value="BA">Bahia</option>
                                                <option value="CE">Ceará</option>
                                                <option value="DF">Distrito Federal</option>
                                                <option value="ES">Espírito Santo</option>
                                                <option value="GO">Goiás</option>
                                                <option value="MA">Maranhão</option>
                                                <option value="MT">Mato Grosso</option>
                                                <option value="MS">Mato Grosso do Sul</option>
                                                <option value="MG">Minas Gerais</option>
                                                <option value="PA">Pará</option>
                                                <option value="PB">Paraíba</option>
                                                <option value="PR">Paraná</option>
                                                <option value="PE">Pernambuco</option>
                                                <option value="PI">Piauí</option>
                                                <option value="RJ">Rio de Janeiro</option>
                                                <option value="RN">Rio Grande do Norte</option>
                                                <option value="RS">Rio Grande do Sul</option>
                                                <option value="RO">Rondônia</option>
                                                <option value="RR">Roraima</option>
                                                <option value="SC">Santa Catarina</option>
                                                <option value="SP">São Paulo</option>
                                                <option value="SE">Sergipe</option>
                                                <option value="TO">Tocantins</option>
                                            </select>
                                        </div>
                                        <div>
                                            <label class="block text-sm font-medium text-gray-700 mb-1">CEP</label>
                                            <input type="text" x-model="newCompany.address_zipcode" placeholder="00000-000"
                                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-purple-500 focus:border-purple-500">
                                        </div>
                                    </div>
                                    
                                    <!-- Configurações -->
                                    <div x-show="activeTab === 'config'" class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                        <div>
                                            <label class="block text-sm font-medium text-gray-700 mb-1">Plano</label>
                                            <select x-model="newCompany.plan_type"
                                                    class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-purple-500 focus:border-purple-500">
                                                <option value="basic">Basic</option>
                                                <option value="professional">Professional</option>
                                                <option value="enterprise">Enterprise</option>
                                            </select>
                                        </div>
                                        <div>
                                            <label class="block text-sm font-medium text-gray-700 mb-1">Máx. Médicos</label>
                                            <input type="number" x-model="newCompany.max_doctors" min="1" max="100"
                                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-purple-500 focus:border-purple-500">
                                        </div>
                                        <div>
                                            <label class="block text-sm font-medium text-gray-700 mb-1">Máx. Usuários</label>
                                            <input type="number" x-model="newCompany.max_users" min="1" max="1000"
                                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-purple-500 focus:border-purple-500">
                                        </div>
                                        <div>
                                            <label class="block text-sm font-medium text-gray-700 mb-1">Máx. Agendamentos/Mês</label>
                                            <input type="number" x-model="newCompany.max_appointments_month" min="100" max="10000"
                                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-purple-500 focus:border-purple-500">
                                        </div>
                                        <div>
                                            <label class="block text-sm font-medium text-gray-700 mb-1">Email do Admin</label>
                                            <input type="email" x-model="newCompany.admin_email"
                                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-purple-500 focus:border-purple-500">
                                        </div>
                                        <div>
                                            <label class="block text-sm font-medium text-gray-700 mb-1">Senha do Admin</label>
                                            <input type="password" x-model="newCompany.admin_password"
                                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-purple-500 focus:border-purple-500">
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="bg-gray-50 px-4 py-3 sm:px-6 sm:flex sm:flex-row-reverse">
                        <button type="submit" :disabled="creating"
                                class="w-full inline-flex justify-center rounded-md border border-transparent shadow-sm px-4 py-2 bg-purple-600 text-base font-medium text-white hover:bg-purple-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-purple-500 sm:ml-3 sm:w-auto sm:text-sm disabled:opacity-50">
                            <span x-show="!creating">Criar Empresa</span>
                            <span x-show="creating" class="flex items-center">
                                <i class="fas fa-spinner fa-spin mr-2"></i>Criando...
                            </span>
                        </button>
                        <button type="button" @click="showCreateCompany = false"
                                class="mt-3 w-full inline-flex justify-center rounded-md border border-gray-300 shadow-sm px-4 py-2 bg-white text-base font-medium text-gray-700 hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 sm:mt-0 sm:ml-3 sm:w-auto sm:text-sm">
                            Cancelar
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script>
        function masterPanel() {
            return {
                showCreateCompany: false,
                creating: false,
                activeTab: 'basic',
                newCompany: {
                    name: '',
                    business_name: '',
                    cnpj: '',
                    business_type: '',
                    contact_email: '',
                    contact_phone: '',
                    whatsapp: '',
                    website: '',
                    address_street: '',
                    address_number: '',
                    address_neighborhood: '',
                    address_city: '',
                    address_state: '',
                    address_zipcode: '',
                    plan_type: 'basic',
                    max_doctors: 5,
                    max_users: 10,
                    max_appointments_month: 1000,
                    admin_email: '',
                    admin_password: 'admin123'
                },

                async createCompany() {
                    this.creating = true;
                    
                    try {
                        const formData = new FormData();
                        formData.append('action', 'create_company');
                        
                        // Adicionar todos os campos
                        Object.keys(this.newCompany).forEach(key => {
                            formData.append(key, this.newCompany[key]);
                        });

                        const response = await fetch('/master.php', {
                            method: 'POST',
                            body: formData
                        });

                        const result = await response.json();

                        if (result.success) {
                            alert(`Empresa criada com sucesso!\n\nEmail do admin: ${result.admin_email}\nSenha: ${result.admin_password}`);
                            this.showCreateCompany = false;
                            this.resetForm();
                            location.reload();
                        } else {
                            alert('Erro: ' + result.message);
                        }
                    } catch (error) {
                        alert('Erro de conexão');
                        console.error('Create company error:', error);
                    } finally {
                        this.creating = false;
                    }
                },

                resetForm() {
                    this.newCompany = {
                        name: '',
                        business_name: '',
                        cnpj: '',
                        business_type: '',
                        contact_email: '',
                        contact_phone: '',
                        whatsapp: '',
                        website: '',
                        address_street: '',
                        address_number: '',
                        address_neighborhood: '',
                        address_city: '',
                        address_state: '',
                        address_zipcode: '',
                        plan_type: 'basic',
                        max_doctors: 5,
                        max_users: 10,
                        max_appointments_month: 1000,
                        admin_email: '',
                        admin_password: 'admin123'
                    };
                    this.activeTab = 'basic';
                },

                async toggleCompanyStatus(companyId, newStatus) {
                    try {
                        const formData = new FormData();
                        formData.append('action', 'toggle_status');
                        formData.append('company_id', companyId);
                        formData.append('new_status', newStatus);

                        const response = await fetch('/master.php', {
                            method: 'POST',
                            body: formData
                        });

                        const result = await response.json();

                        if (result.success) {
                            alert(result.message);
                            location.reload();
                        } else {
                            alert('Erro: ' + result.message);
                        }
                    } catch (error) {
                        alert('Erro de conexão');
                        console.error('Toggle status error:', error);
                    }
                },

                async logout() {
                    try {
                        const formData = new FormData();
                        formData.append('action', 'logout');
                        
                        const response = await fetch('/auth.php', {
                            method: 'POST',
                            body: formData
                        });
                        
                        const result = await response.json();
                        
                        if (result.success) {
                            window.location.href = '/login.php';
                        }
                    } catch (error) {
                        console.error('Logout error:', error);
                        window.location.href = '/login.php';
                    }
                }
            }
        }
    </script>

</body>
</html>
