<?php
/**
 * Gerenciamento de Usuários
 * - Listar usuários de todas as empresas (Master)
 * - Criar usuários para empresas específicas
 * - Editar/ativar/desativar usuários
 */

require_once __DIR__ . '/auth.php';

// Verificar se é MASTER
requireMaster();

$user = getCurrentUser();

// ===================== FUNÇÕES DE USUÁRIOS =====================

function getAllUsers() {
    $pdo = getPdo();
    $stmt = $pdo->query("
        SELECT u.id, u.email, u.username, u.first_name, u.last_name, u.role, u.is_active, u.last_login, u.created_at,
               c.name as company_name, c.business_type
        FROM users u
        LEFT JOIN companies c ON u.company_id = c.id
        ORDER BY u.created_at DESC
    ");
    return $stmt->fetchAll();
}

function getAllCompanies() {
    $pdo = getPdo();
    $stmt = $pdo->query("SELECT id, name, business_type FROM companies WHERE status = 'active' ORDER BY name");
    return $stmt->fetchAll();
}

function createUser($companyId, $email, $username, $password, $firstName, $lastName, $role) {
    $pdo = getPdo();
    
    try {
        // Verificar se email já existe
        $stmt = $pdo->prepare("SELECT id FROM users WHERE email = ?");
        $stmt->execute([$email]);
        if ($stmt->fetch()) {
            return ['success' => false, 'message' => 'Email já cadastrado'];
        }
        
        // Verificar se username já existe na empresa
        if ($username) {
            $stmt = $pdo->prepare("SELECT id FROM users WHERE company_id = ? AND username = ?");
            $stmt->execute([$companyId, $username]);
            if ($stmt->fetch()) {
                return ['success' => false, 'message' => 'Username já existe nesta empresa'];
            }
        }
        
        $passwordHash = password_hash($password, PASSWORD_DEFAULT);
        
        $stmt = $pdo->prepare("
            INSERT INTO users (company_id, email, username, password_hash, first_name, last_name, role, is_active, created_by)
            VALUES (?, ?, ?, ?, ?, ?, ?, 1, ?)
        ");
        
        $stmt->execute([$companyId, $email, $username, $passwordHash, $firstName, $lastName, $role, $user['id']]);
        
        return [
            'success' => true,
            'user_id' => $pdo->lastInsertId(),
            'email' => $email,
            'password' => $password
        ];
        
    } catch (Exception $e) {
        return ['success' => false, 'message' => $e->getMessage()];
    }
}

function toggleUserStatus($userId) {
    $pdo = getPdo();
    
    try {
        $stmt = $pdo->prepare("SELECT is_active FROM users WHERE id = ?");
        $stmt->execute([$userId]);
        $user = $stmt->fetch();
        
        if (!$user) {
            return ['success' => false, 'message' => 'Usuário não encontrado'];
        }
        
        $newStatus = $user['is_active'] ? 0 : 1;
        
        $stmt = $pdo->prepare("UPDATE users SET is_active = ? WHERE id = ?");
        $stmt->execute([$newStatus, $userId]);
        
        return [
            'success' => true,
            'new_status' => $newStatus,
            'message' => $newStatus ? 'Usuário ativado' : 'Usuário desativado'
        ];
        
    } catch (Exception $e) {
        return ['success' => false, 'message' => $e->getMessage()];
    }
}

// Processar requests
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    switch ($action) {
        case 'create_user':
            $companyId = $_POST['company_id'] ?? '';
            $email = $_POST['email'] ?? '';
            $username = $_POST['username'] ?? '';
            $password = $_POST['password'] ?? '';
            $firstName = $_POST['first_name'] ?? '';
            $lastName = $_POST['last_name'] ?? '';
            $role = $_POST['role'] ?? 'user';
            
            if (empty($companyId) || empty($email) || empty($password)) {
                echo json_encode(['success' => false, 'message' => 'Empresa, email e senha são obrigatórios']);
                exit;
            }
            
            $result = createUser($companyId, $email, $username, $password, $firstName, $lastName, $role);
            echo json_encode($result);
            exit;
            
        case 'toggle_status':
            $userId = $_POST['user_id'] ?? '';
            $result = toggleUserStatus($userId);
            echo json_encode($result);
            exit;
    }
}

$users = getAllUsers();
$companies = getAllCompanies();

?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Usuários - Master Panel</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script src="https://unpkg.com/alpinejs@3.x.x/dist/cdn.min.js" defer></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
</head>
<body class="bg-gray-50" x-data="usersPanel()">
    
    <!-- Sidebar -->
    <div class="fixed inset-y-0 left-0 z-50 w-64 bg-white shadow-lg">
        <!-- Logo -->
        <div class="flex items-center justify-center h-16 bg-gradient-to-r from-purple-600 to-indigo-600">
            <i class="fas fa-crown text-white text-2xl mr-2"></i>
            <span class="text-white font-bold text-xl">Master Panel</span>
        </div>

        <!-- Navigation -->
        <nav class="mt-8">
            <div class="px-4 mb-4">
                <p class="text-xs font-semibold text-gray-500 uppercase tracking-wider">Master</p>
            </div>
            
            <a href="/master.php" class="flex items-center px-4 py-3 text-gray-700 hover:bg-gray-100">
                <i class="fas fa-tachometer-alt mr-3"></i>
                Dashboard Master
            </a>
            
            <a href="/users.php" class="flex items-center px-4 py-3 text-purple-600 bg-purple-50 border-r-2 border-purple-600">
                <i class="fas fa-users mr-3"></i>
                Usuários
            </a>
            
            <a href="/companies.php" class="flex items-center px-4 py-3 text-gray-700 hover:bg-gray-100">
                <i class="fas fa-building mr-3"></i>
                Empresas
            </a>
            
            <a href="/analytics.php" class="flex items-center px-4 py-3 text-gray-700 hover:bg-gray-100">
                <i class="fas fa-chart-bar mr-3"></i>
                Analytics
            </a>
            
            <a href="/settings.php" class="flex items-center px-4 py-3 text-gray-700 hover:bg-gray-100">
                <i class="fas fa-cog mr-3"></i>
                Configurações
            </a>
        </nav>

        <!-- User Info -->
        <div class="absolute bottom-0 w-full p-4 border-t">
            <div class="flex items-center">
                <div class="w-8 h-8 bg-purple-600 rounded-full flex items-center justify-center">
                    <i class="fas fa-crown text-white text-sm"></i>
                </div>
                <div class="ml-3">
                    <p class="text-sm font-medium text-gray-900"><?= htmlspecialchars($user['email']) ?></p>
                    <p class="text-xs text-gray-500">Master Admin</p>
                </div>
            </div>
            <button @click="logout()" class="w-full mt-3 text-left text-sm text-gray-600 hover:text-gray-900">
                <i class="fas fa-sign-out-alt mr-2"></i>Sair
            </button>
        </div>
    </div>

    <!-- Main Content -->
    <div class="ml-64">
        <!-- Top Bar -->
        <header class="bg-white shadow-sm border-b">
            <div class="flex items-center justify-between px-6 py-4">
                <div>
                    <h1 class="text-2xl font-bold text-gray-900">Gerenciar Usuários</h1>
                    <p class="text-gray-600">Criar e gerenciar usuários de todas as empresas</p>
                </div>
                
                <div class="flex items-center space-x-4">
                    <button @click="showCreateUser = true" 
                            class="bg-gradient-to-r from-purple-600 to-indigo-600 text-white px-6 py-2 rounded-lg hover:from-purple-700 hover:to-indigo-700 transition-all duration-200 shadow-lg">
                        <i class="fas fa-plus mr-2"></i>Novo Usuário
                    </button>
                </div>
            </div>
        </header>

        <!-- Content -->
        <main class="p-6">
            <!-- Stats -->
            <div class="grid grid-cols-1 md:grid-cols-4 gap-6 mb-8">
                <div class="bg-white rounded-lg shadow-lg p-6 border-l-4 border-blue-500">
                    <div class="flex items-center">
                        <div class="p-3 rounded-full bg-blue-100 text-blue-600">
                            <i class="fas fa-users text-xl"></i>
                        </div>
                        <div class="ml-4">
                            <p class="text-sm font-medium text-gray-600">Total Usuários</p>
                            <p class="text-2xl font-semibold text-gray-900"><?= count($users) ?></p>
                        </div>
                    </div>
                </div>
                
                <div class="bg-white rounded-lg shadow-lg p-6 border-l-4 border-green-500">
                    <div class="flex items-center">
                        <div class="p-3 rounded-full bg-green-100 text-green-600">
                            <i class="fas fa-user-check text-xl"></i>
                        </div>
                        <div class="ml-4">
                            <p class="text-sm font-medium text-gray-600">Ativos</p>
                            <p class="text-2xl font-semibold text-gray-900"><?= count(array_filter($users, fn($u) => $u['is_active'])) ?></p>
                        </div>
                    </div>
                </div>
                
                <div class="bg-white rounded-lg shadow-lg p-6 border-l-4 border-purple-500">
                    <div class="flex items-center">
                        <div class="p-3 rounded-full bg-purple-100 text-purple-600">
                            <i class="fas fa-crown text-xl"></i>
                        </div>
                        <div class="ml-4">
                            <p class="text-sm font-medium text-gray-600">Admins</p>
                            <p class="text-2xl font-semibold text-gray-900"><?= count(array_filter($users, fn($u) => $u['role'] === 'admin')) ?></p>
                        </div>
                    </div>
                </div>
                
                <div class="bg-white rounded-lg shadow-lg p-6 border-l-4 border-yellow-500">
                    <div class="flex items-center">
                        <div class="p-3 rounded-full bg-yellow-100 text-yellow-600">
                            <i class="fas fa-clock text-xl"></i>
                        </div>
                        <div class="ml-4">
                            <p class="text-sm font-medium text-gray-600">Online Hoje</p>
                            <p class="text-2xl font-semibold text-gray-900"><?= count(array_filter($users, fn($u) => $u['last_login'] && date('Y-m-d', strtotime($u['last_login'])) === date('Y-m-d'))) ?></p>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Users Table -->
            <div class="bg-white rounded-lg shadow-lg">
                <div class="px-6 py-4 border-b border-gray-200 bg-gradient-to-r from-gray-50 to-gray-100">
                    <div class="flex items-center justify-between">
                        <h3 class="text-lg font-medium text-gray-900">Lista de Usuários</h3>
                        <div class="flex items-center space-x-2">
                            <span class="text-sm text-gray-500">Total: <?= count($users) ?></span>
                            <button @click="refreshUsers()" class="text-gray-400 hover:text-gray-600">
                                <i class="fas fa-sync-alt"></i>
                            </button>
                        </div>
                    </div>
                </div>
                <div class="overflow-x-auto">
                    <table class="min-w-full divide-y divide-gray-200">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Usuário</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Empresa</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Role</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Último Login</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Ações</th>
                            </tr>
                        </thead>
                        <tbody class="bg-white divide-y divide-gray-200">
                            <?php foreach ($users as $u): ?>
                            <tr class="hover:bg-gray-50">
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <div class="flex items-center">
                                        <div class="w-10 h-10 bg-gradient-to-r from-blue-500 to-purple-500 rounded-full flex items-center justify-center text-white font-bold">
                                            <?= strtoupper(substr($u['first_name'] ?: $u['email'], 0, 2)) ?>
                                        </div>
                                        <div class="ml-4">
                                            <div class="text-sm font-medium text-gray-900">
                                                <?= htmlspecialchars($u['first_name'] . ' ' . $u['last_name'] ?: $u['email']) ?>
                                            </div>
                                            <div class="text-sm text-gray-500"><?= htmlspecialchars($u['email']) ?></div>
                                            <?php if ($u['username']): ?>
                                            <div class="text-xs text-gray-400">@<?= htmlspecialchars($u['username']) ?></div>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <div class="text-sm text-gray-900">
                                        <?= $u['company_name'] ? htmlspecialchars($u['company_name']) : '<span class="text-purple-600 font-medium">Master</span>' ?>
                                    </div>
                                    <?php if ($u['business_type']): ?>
                                    <div class="text-xs text-gray-500"><?= htmlspecialchars($u['business_type']) ?></div>
                                    <?php endif; ?>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <span class="px-2 py-1 text-xs font-medium rounded-full
                                        <?php 
                                        switch($u['role']) {
                                            case 'master': echo 'bg-purple-100 text-purple-800'; break;
                                            case 'admin': echo 'bg-blue-100 text-blue-800'; break;
                                            case 'manager': echo 'bg-green-100 text-green-800'; break;
                                            case 'user': echo 'bg-gray-100 text-gray-800'; break;
                                            case 'receptionist': echo 'bg-yellow-100 text-yellow-800'; break;
                                        }
                                        ?>">
                                        <?= ucfirst($u['role']) ?>
                                    </span>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <span class="px-2 py-1 text-xs font-medium rounded-full
                                        <?= $u['is_active'] ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800' ?>">
                                        <?= $u['is_active'] ? 'Ativo' : 'Inativo' ?>
                                    </span>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                    <?php if ($u['last_login']): ?>
                                        <?= date('d/m/Y H:i', strtotime($u['last_login'])) ?>
                                    <?php else: ?>
                                        <span class="text-gray-400">Nunca</span>
                                    <?php endif; ?>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                    <div class="flex items-center space-x-2">
                                        <button @click="toggleStatus(<?= $u['id'] ?>, <?= $u['is_active'] ? 'false' : 'true' ?>)"
                                                class="text-blue-600 hover:text-blue-900" title="Alterar status">
                                            <i class="fas fa-<?= $u['is_active'] ? 'pause' : 'play' ?>"></i>
                                        </button>
                                        <button @click="editUser(<?= $u['id'] ?>)"
                                                class="text-green-600 hover:text-green-900" title="Editar">
                                            <i class="fas fa-edit"></i>
                                        </button>
                                        <button @click="resetPassword(<?= $u['id'] ?>)"
                                                class="text-yellow-600 hover:text-yellow-900" title="Resetar senha">
                                            <i class="fas fa-key"></i>
                                        </button>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </main>
    </div>

    <!-- Modal Criar Usuário -->
    <div x-show="showCreateUser" 
         x-transition:enter="transition-opacity ease-linear duration-300"
         x-transition:enter-start="opacity-0"
         x-transition:enter-end="opacity-100"
         x-transition:leave="transition-opacity ease-linear duration-300"
         x-transition:leave-start="opacity-100"
         x-transition:leave-end="opacity-0"
         class="fixed inset-0 z-50 overflow-y-auto" 
         style="display: none;">
        
        <div class="flex items-center justify-center min-h-screen pt-4 px-4 pb-20 text-center sm:block sm:p-0">
            <div class="fixed inset-0 bg-gray-500 bg-opacity-75 transition-opacity" @click="showCreateUser = false"></div>

            <div class="inline-block align-bottom bg-white rounded-lg text-left overflow-hidden shadow-xl transform transition-all sm:my-8 sm:align-middle sm:max-w-lg sm:w-full">
                <form @submit.prevent="createUser">
                    <div class="bg-white px-4 pt-5 pb-4 sm:p-6 sm:pb-4">
                        <div class="sm:flex sm:items-start">
                            <div class="mx-auto flex-shrink-0 flex items-center justify-center h-12 w-12 rounded-full bg-purple-100 sm:mx-0 sm:h-10 sm:w-10">
                                <i class="fas fa-user-plus text-purple-600"></i>
                            </div>
                            <div class="mt-3 text-center sm:mt-0 sm:ml-4 sm:text-left w-full">
                                <h3 class="text-lg leading-6 font-medium text-gray-900 mb-4">Novo Usuário</h3>
                                
                                <div class="space-y-4">
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-1">Empresa *</label>
                                        <select x-model="newUser.company_id" required
                                                class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-purple-500 focus:border-purple-500">
                                            <option value="">Selecione a empresa...</option>
                                            <?php foreach ($companies as $company): ?>
                                            <option value="<?= $company['id'] ?>"><?= htmlspecialchars($company['name']) ?> (<?= htmlspecialchars($company['business_type']) ?>)</option>
                                            <?php endforeach; ?>
                                        </select>
                                    </div>
                                    
                                    <div class="grid grid-cols-2 gap-4">
                                        <div>
                                            <label class="block text-sm font-medium text-gray-700 mb-1">Nome</label>
                                            <input type="text" x-model="newUser.first_name"
                                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-purple-500 focus:border-purple-500">
                                        </div>
                                        <div>
                                            <label class="block text-sm font-medium text-gray-700 mb-1">Sobrenome</label>
                                            <input type="text" x-model="newUser.last_name"
                                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-purple-500 focus:border-purple-500">
                                        </div>
                                    </div>
                                    
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-1">Email *</label>
                                        <input type="email" x-model="newUser.email" required
                                               class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-purple-500 focus:border-purple-500">
                                    </div>
                                    
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-1">Username</label>
                                        <input type="text" x-model="newUser.username"
                                               class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-purple-500 focus:border-purple-500">
                                    </div>
                                    
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-1">Senha *</label>
                                        <input type="password" x-model="newUser.password" required
                                               class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-purple-500 focus:border-purple-500">
                                    </div>
                                    
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-1">Role</label>
                                        <select x-model="newUser.role" required
                                                class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-purple-500 focus:border-purple-500">
                                            <option value="user">User</option>
                                            <option value="receptionist">Receptionist</option>
                                            <option value="manager">Manager</option>
                                            <option value="admin">Admin</option>
                                        </select>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="bg-gray-50 px-4 py-3 sm:px-6 sm:flex sm:flex-row-reverse">
                        <button type="submit" :disabled="creating"
                                class="w-full inline-flex justify-center rounded-md border border-transparent shadow-sm px-4 py-2 bg-purple-600 text-base font-medium text-white hover:bg-purple-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-purple-500 sm:ml-3 sm:w-auto sm:text-sm disabled:opacity-50">
                            <span x-show="!creating">Criar Usuário</span>
                            <span x-show="creating" class="flex items-center">
                                <i class="fas fa-spinner fa-spin mr-2"></i>Criando...
                            </span>
                        </button>
                        <button type="button" @click="showCreateUser = false"
                                class="mt-3 w-full inline-flex justify-center rounded-md border border-gray-300 shadow-sm px-4 py-2 bg-white text-base font-medium text-gray-700 hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 sm:mt-0 sm:ml-3 sm:w-auto sm:text-sm">
                            Cancelar
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script>
        function usersPanel() {
            return {
                showCreateUser: false,
                creating: false,
                newUser: {
                    company_id: '',
                    email: '',
                    username: '',
                    password: '',
                    first_name: '',
                    last_name: '',
                    role: 'user'
                },

                async createUser() {
                    this.creating = true;
                    
                    try {
                        const formData = new FormData();
                        formData.append('action', 'create_user');
                        formData.append('company_id', this.newUser.company_id);
                        formData.append('email', this.newUser.email);
                        formData.append('username', this.newUser.username);
                        formData.append('password', this.newUser.password);
                        formData.append('first_name', this.newUser.first_name);
                        formData.append('last_name', this.newUser.last_name);
                        formData.append('role', this.newUser.role);

                        const response = await fetch('/users.php', {
                            method: 'POST',
                            body: formData
                        });

                        const result = await response.json();

                        if (result.success) {
                            alert(`Usuário criado com sucesso!\n\nEmail: ${result.email}\nSenha: ${result.password}`);
                            this.showCreateUser = false;
                            this.resetForm();
                            location.reload();
                        } else {
                            alert('Erro: ' + result.message);
                        }
                    } catch (error) {
                        alert('Erro de conexão');
                        console.error('Create user error:', error);
                    } finally {
                        this.creating = false;
                    }
                },

                resetForm() {
                    this.newUser = {
                        company_id: '',
                        email: '',
                        username: '',
                        password: '',
                        first_name: '',
                        last_name: '',
                        role: 'user'
                    };
                },

                async toggleStatus(userId, newStatus) {
                    try {
                        const formData = new FormData();
                        formData.append('action', 'toggle_status');
                        formData.append('user_id', userId);

                        const response = await fetch('/users.php', {
                            method: 'POST',
                            body: formData
                        });

                        const result = await response.json();

                        if (result.success) {
                            alert(result.message);
                            location.reload();
                        } else {
                            alert('Erro: ' + result.message);
                        }
                    } catch (error) {
                        alert('Erro de conexão');
                        console.error('Toggle status error:', error);
                    }
                },

                async logout() {
                    try {
                        const formData = new FormData();
                        formData.append('action', 'logout');
                        
                        const response = await fetch('/auth.php', {
                            method: 'POST',
                            body: formData
                        });
                        
                        const result = await response.json();
                        
                        if (result.success) {
                            window.location.href = '/login.php';
                        }
                    } catch (error) {
                        console.error('Logout error:', error);
                        window.location.href = '/login.php';
                    }
                }
            }
        }
    </script>

</body>
</html>
